/*
 * Copyright (c) 2016 RedBear
 * Copyright (c) 2017 Fredrik Hubinette
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */
#include <nRF5x_BLE_API.h>

const char version[] = "$Id: NanoUart.ino,v 1.5 2018/11/07 04:05:32 hubbe Exp $";

#define TXRX_BUF_LEN  20

char password[21];
char device_name[240];
char short_device_name[21];

enum AuthenticationState {
  AUTH_NO,
  AUTH_WRONG_PASSWORD,
  AUTH_PENDING,
  AUTH_YES
};
AuthenticationState authenticated = AUTH_NO;

#ifndef COMMON_STATE_MACHINE_H
#define COMMON_STATE_MACHINE_H

/*
 * State machine code.
 * This code uses the fact that switch can jump into code to create
 * lightweight pseudo-thread. These threads have a lot of limitations,
 * but they can make code much easier to read compared to manually
 * written state machines. Lots of examples below.
 */
// Note, you cannot have two YIELD() on the same line.
#define YIELD() do { state_machine_.next_state_ = __LINE__; return; case __LINE__: break; } while(0)
#define SLEEP(MILLIS) do { state_machine_.sleep_until_ = millis() + (MILLIS); while (millis() < state_machine_.sleep_until_) YIELD(); } while(0)
#define SLEEP_MICROS(MICROS) do { state_machine_.sleep_until_ = micros() + (MICROS); while (micros() < state_machine_.sleep_until_) YIELD(); } while(0)
#define STATE_MACHINE_BEGIN() switch(state_machine_.next_state_) { case -1:
#define STATE_MACHINE_END() state_machine_.next_state_ = -2; case -2: break; } 

#define CALL(SM, FUN) do {                  \
  state_machine_.next_state_ = __LINE__;    \
  case __LINE__:                            \
  FUN();                                    \
  if (SM.next_state == -2) return;          \
  SM.reset_state_machine();                 \
} while(0)

struct StateMachineState {
  int next_state_;
  uint32_t sleep_until_;
  void reset_state_machine() {
    next_state_ = -1;
  }
};

class StateMachine {
protected:
  StateMachineState state_machine_;
};

#endif

class Blinky : public StateMachine {
public:
  Blinky() {
    state_machine_.reset_state_machine();
  }
  void Loop() {
    STATE_MACHINE_BEGIN();
    // Three blinks means we're working.
    for (i = 0; i < 5; i++) {
      analogWrite(13, 128);
      SLEEP(300);
      analogWrite(13, 0);
      SLEEP(300);
    }

    while (true) {
      SLEEP(20);

      if (activity_) {
        activity_ = false;
        analogWrite(13, 0xff);
	continue;
      }
      
      switch (authenticated) {
        case AUTH_PENDING: {
          // Fast fade in/out while authenticating
          uint32_t x = millis();
          if (x & 0x100) {
            x = 0xff - (x & 0xff);
          } else {
            x = (x & 0xff);
          }
          analogWrite(13, x);
          break;
        }
        case AUTH_YES: {
          // Fade in/out, but slower and less bright
          uint32_t x = (millis() >> 2);
          if (x & 0x100) {
            x = 0xff - (x & 0xff);
          } else {
            x = (x & 0xff);
          }
          analogWrite(13, x >> 2);
          break;
        }

        case AUTH_NO:
        case AUTH_WRONG_PASSWORD:
          if ((millis() & 2047) < 10) {
            analogWrite(13, 64);
          } else {
            analogWrite(13, 0);
          }
          break;
      }
    }

    STATE_MACHINE_END();
  }

  void Activity() { activity_ = true; }
private:
  bool activity_ = false;
  int i;
};

Blinky blinky;


BLE ble;

#define UUID16 0x71, 0x3D
#define UUID96 0x38, 0x9c, 0xf6, 0x37, 0xb1, 0xd7, 0x91, 0xb3, 0x61, 0xae, 0x76, 0x78

// The uuid of service and characteristics
static const uint8_t service1_uuid[]        = {UUID16, 0, 0, UUID96};
static const uint8_t service1_tx_uuid[]     = {UUID16, 0, 3, UUID96};
static const uint8_t service1_rx_uuid[]     = {UUID16, 0, 2, UUID96};
static const uint8_t service1_pw_uuid[]     = {UUID16, 0, 4, UUID96};
static const uint8_t service1_status_uuid[] = {UUID16, 0, 5, UUID96};
static const uint8_t service1_uuid_rev[]   = {
  0x78, 0x76, 0xae, 0x61,  0xb3, 0x91, 0xd7, 0xb1,
  0x37, 0xf6, 0x9c, 0x38,     0,    0, 0x3D, 0x71 };

template<int N>
class Buffer {
public:
  Buffer() : start_(0), end_(0) {}
  size_t buffered() const { return end_ - start_; };
  size_t available() const { return sizeof(buffer_) - buffered(); }
  void write(uint8_t c) {
    size_t end_pos = end_ & (N - 1);
    buffer_[end_pos] = c;
    end_++;
  }
  size_t read(uint8_t* dest, size_t bytes) {
    size_t ret = bytes = min(bytes, buffered());
    size_t start = start_;
    while (bytes) {
      size_t start_pos = start &  (N - 1);
      size_t to_copy = min(N - start_pos, bytes);
      memcpy(dest, buffer_ + start_pos, to_copy);
      bytes -= to_copy;
      dest += to_copy;
      start += to_copy;
    }
    return ret;
  }
  void pop(size_t bytes) {
    start_ += bytes;
  }

  int getc() {
    uint8_t tmp;
    size_t bytes = read(&tmp, 1);
    if (bytes == 0) return -1;
    pop(1);
    return tmp;
  }

  void clear() { pop(buffered()); }
  
private:
  volatile size_t start_;
  volatile size_t end_;
  uint8_t buffer_[N];
};

Buffer<16384> rx_buffer;

uint8_t tx_value[TXRX_BUF_LEN] = {0,};
uint8_t rx_value[TXRX_BUF_LEN] = {0,};
uint8_t pw_value[TXRX_BUF_LEN] = {0,};
uint8_t status_value[TXRX_BUF_LEN] = {0,};

// Create characteristic and service
GattCharacteristic  characteristic1(
  service1_tx_uuid, tx_value, 1, TXRX_BUF_LEN,
  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE );

GattCharacteristic  characteristic2(
  service1_rx_uuid, rx_value, 1, TXRX_BUF_LEN,
  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);

GattCharacteristic  characteristic3(
  service1_pw_uuid, pw_value, 1, TXRX_BUF_LEN,
  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE );

GattCharacteristic  characteristic4(
  service1_status_uuid, status_value, 1, TXRX_BUF_LEN,
  GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);


GattCharacteristic *uartChars[] = {
  &characteristic1,
  &characteristic2,
  &characteristic3,
  &characteristic4
};

GattService uartService(
  service1_uuid, uartChars,
  sizeof(uartChars) / sizeof(GattCharacteristic *));


void disconnectionCallBack(const Gap::DisconnectionCallbackParams_t *params) {
  Serial.println("# Disconnected\n");
  authenticated = AUTH_NO;
  ble.startAdvertising();
}

void gattServerWriteCallBack(const GattWriteCallbackParams *Handler) {
  // TODO: Replace cleartext password with challenge-response.
  if (Handler->handle == characteristic3.getValueAttribute().getHandle()) {
    if (Handler->len == strlen(password) &&
        memcmp(Handler->data, password, Handler->len) == 0) {
      authenticated = AUTH_PENDING;
    } else {
      authenticated = AUTH_WRONG_PASSWORD;
    }
  }

  if (Handler->handle == characteristic1.getValueAttribute().getHandle() &&
      authenticated == AUTH_YES) {
    blinky.Activity();
    for(uint32_t index=0; index < Handler->len; index++) {
      Serial.write(Handler->data[index]);
    }
  }
}

uint32_t last_rx = millis();

void uart_handle(uint32_t id, SerialIrq event) {
  // Serial rx IRQ
  last_rx = millis();
  while (Serial.available() && rx_buffer.available()) {
    rx_buffer.write(Serial.read());
  }
}

// Read a line from rx_buffer with timeout.
bool ReadLine(char* buffer, size_t length, uint32_t timeout) {
  uint32_t start = millis();
  while (length && millis() - start < timeout) {
    int c = rx_buffer.getc();
    if (c == -1) continue;
    if (c == '\n') {
      *buffer = 0;
      return true;
    }
    *buffer = c;
    buffer++;
    length--;
  }
  return false;
}

void InitSerial() {
  Serial.begin(115200);
  Serial.attach(uart_handle);
#if 0
  // For debugging
  strcpy(password, "password");
  strcpy(device_name, "TeensySaber");
  strcpy(short_device_name, "Saber");
#else
  uint32_t got = 0;
  while (got != 7) {
    got = 0;
    char line[128];
    delay(1000);
    rx_buffer.clear();
    uint32_t start = millis();
    Serial.println("get_ble_config");
    if (!ReadLine(line, sizeof(line), 1000)) continue;
    if (strcmp(line, "-+=BEGIN_OUTPUT=+-")) continue;
    while (millis() - start < 5000) {
      if (!ReadLine(line, sizeof(line), 1000)) continue;
      if (line[0] == '-') break;
      line[127] = 0;
      if (!strcmp(line, "-+=END_OUTPUT=+-")) break;
      char* eq = strchr(line, '=');
      if (!eq) break;
      *eq = 0;
      char *value = eq + 1;
      if (!strcmp(line, "name")) {
        strncpy(device_name, value, sizeof(device_name));
        got |= 1;
      } else if (!strcmp(line, "shortname")) {
        strncpy(short_device_name, value, sizeof(short_device_name));
        got |= 2;
      } else if (!strcmp(line, "password")) {
        strncpy(password, value, sizeof(password));
        got |= 4;
      }
    }
  }
#endif
  Serial.print("# BLE Configured ");
  Serial.println(version);
}

void setup() {
  analogWrite(13, 255);
  delay(100);
  analogWrite(13, 0);
  InitSerial();
  
  ble.init();
  ble.onDisconnection(disconnectionCallBack);
  ble.onDataWritten(gattServerWriteCallBack);

  // setup adv_data and srp_data
  ble.accumulateAdvertisingPayload(
    GapAdvertisingData::BREDR_NOT_SUPPORTED |
    GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
  ble.accumulateAdvertisingPayload(
    GapAdvertisingData::SHORTENED_LOCAL_NAME,
    (uint8_t *)short_device_name,
    strlen(short_device_name));
  ble.accumulateAdvertisingPayload(
    GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
    (const uint8_t *)service1_uuid_rev,
    sizeof(service1_uuid_rev));
  // set adv_type
  ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
  // add service
  ble.addService(uartService);
  // set device name
  ble.setDeviceName((uint8_t*)device_name);
  // set tx power,valid values are -40, -20, -16, -12, -8, -4, 0, 4
  ble.setTxPower(4);
  // set adv_interval, 100ms in multiples of 0.625ms.
  ble.setAdvertisingInterval(160);
  // set adv_timeout, in seconds
  ble.setAdvertisingTimeout(0);
  // start advertising
  ble.startAdvertising();
}

void loop() {
  ble.processEvents();
  blinky.Loop();
  // put your main code here, to run repeatedly:

  switch (authenticated) {
    case AUTH_PENDING:
      Serial.print("# BLE Authenticated ");
      Serial.println(
      ble.updateCharacteristicValue(
        characteristic4.getValueAttribute().getHandle(),
        (const uint8_t*)"OK", 2));
      authenticated = AUTH_YES;
      // Fall through.
    case AUTH_YES:
      break;

    case AUTH_WRONG_PASSWORD:
      Serial.print("# BLE Wrong Password ");
      Serial.println(
      ble.updateCharacteristicValue(
        characteristic4.getValueAttribute().getHandle(),
        (const uint8_t*)"WRONG_PW", 8));
      authenticated = AUTH_NO;
      // Fall through.
    case AUTH_NO:
      rx_buffer.clear();
      return;
  }
    
  if (rx_buffer.buffered()) {
    if (rx_buffer.buffered() >= TXRX_BUF_LEN || millis() - last_rx > 50) {
      uint8_t buf[TXRX_BUF_LEN];
      uint32_t bytes = rx_buffer.read(buf, sizeof(buf));
      switch (ble.updateCharacteristicValue(
                characteristic2.getValueAttribute().getHandle(), buf, bytes)) {
        case BLE_ERROR_NONE:
          // OK
          blinky.Activity();
          rx_buffer.pop(bytes);
          break;

        case BLE_ERROR_BUFFER_OVERFLOW:
        case BLE_STACK_BUSY:
        case BLE_ERROR_NO_MEM:
          // Try again later;
          break;
        default:
          // Fail
          break;
      }
    }
  }
}
